<?php
namespace addons\dbassistant\lib;

use app\common\addons\AddonsLib;
use think\facade\Db;

/**
 * 数据库备份实现类
 */
class Dbackup extends AddonsLib
{
    /**
     * 备份数据的时候，打开一个卷，用于写入数据
     * @param array $file 文件数组
     */
    private function open(&$file)
    {
        if($file['fp']) {
            return true;
        }

        if (is_dir($file['path']) == false) {//如果路径不存在，则创建
            @mkdir($file['path'], 0777, true);
        }
        if(substr($file['path'], -1) != DIRECTORY_SEPARATOR){
            $file['path'] .= DIRECTORY_SEPARATOR;
        }

        if($file['name'] != ""){
            $filename = "{$file['name']}-{$file['part']}.sql"; 
        }else{
            $filename = "{$file['part']}.sql";
        }
        $file['fp'] = @fopen($file['path'] . $filename, 'a');
        
        if(!$file['fp']){
            return false;
        }
        //写初始化
        $this->initInfo($file);
        return true;
    }

    /**
     * 关闭文件句柄
     */
    private function close(&$file){
        if($file['fp']){
            @fclose($file['fp']);    
        }
        $file['fp'] = null;
        return true;
    }
 
    /**
     * 写入初始数据
     * @return mixed
     */
    private function initInfo(&$file)
    {
        $sql  = "-- -----------------------------\n";
        $sql .= "-- MySQL Data Transfer\n";
        $sql .= "--\n";
        $sql .= "-- Host     : " . config('database.connections.mysql.hostname', 'null') . "\n";
        $sql .= "-- Port     : " . config('database.connections.mysql.hostport', 'null') . "\n";
        $sql .= "-- Database : " . config('database.connections.mysql.database', 'null') . "\n";
        $sql .= "--\n";
        if($file['size_limit'] > 0){
            $sql .= "-- Part : #{$file['part']}\n";
        }
        $sql .= "-- Date : " . date("Y-m-d H:i:s") . "\n";
        $sql .= "-- -----------------------------\n\n";
        $sql .= "SET FOREIGN_KEY_CHECKS = 0;\n\n";
        return $this->write($file, $sql);
    }
 
    /**
     * 写入SQL语句
     * @param string $sql 要写入的SQL语句
     * @return int
     */
    private function write(&$file, $sql)
    {
        $size = strlen($sql);

        if($size <= 0){
            return recode(10000);
        }

        if(!$this->open($file)){
            return recode(10100, '', '文件打开失败');
        }
        if(!$file['fp']){
            return recode(10100, $sql);
        }
        $re = @fwrite($file['fp'], $sql);
        if(!$re){
            $this->close($file);
            return recode(10100, '', '写文件失败');
        }
        $file['size'] += $size;
        $file['allsize'] += $size;

        //判断文件是否满了
        if($file['size_limit'] > 0 && $file['size'] >= $file['size_limit']){
            //释放句柄
            if($this->close($file)){
                $file['part']++;
                $file['size'] = 0;
            }else{
                return recode(10100, '', '关闭文件失败');
            }
        }

        return recode(10000);
    }

    //保存表结构
    private function tableStructure(&$file, $table){
        $result = Db::query("SHOW CREATE TABLE `{$table}`");
        $result = array_map('array_change_key_case', $result);

        $sql  = "";
        $sql .= "-- -----------------------------\n";
        $sql .= "-- Table structure for `{$table}`\n";
        $sql .= "-- -----------------------------\n";
        //$sql .= "DROP TABLE IF EXISTS `{$table}`;\n";
        $sql .= trim(str_replace('CREATE TABLE', 'CREATE TABLE IF NOT EXISTS', $result[0]['create table'])) . ";\n\n";
        
        return $this->write($file, $sql);
    }

    //保存表数据
    private function tableData(&$file, $table){
        $sql  = "-- -----------------------------\n";
        $sql .= "-- Records of `{$table}`\n";
        $sql .= "-- -----------------------------\n";
        $sql .= "TRUNCATE `{$table}`;\n";
        $re = $this->write($file, $sql);
        if($re['code'] >= 10100){
            return $re;
        }

        $limit = 1000;      //一次处理数量
        $start = 0;

        do {
            $result = Db::query("SELECT * FROM `{$table}` LIMIT {$start}, {$limit}");
            if(count($result) > 0){
                $sql = "INSERT INTO `{$table}` ".$this->getHeader($result[0])." VALUES " . $this->createRows($result) . ";\n";
                $re = $this->write($file, $sql);
                if ($re['code'] >= 10100) {
                    return $re;
                }
            }
            if(count($result) < $limit){
                break;
            }
            $start += $limit;
        }while(true);

        return recode(10000);
    }

    //获取导入sql头部
    private function getHeader($row){
        return '(' . implode(',',array_map(function($i){return '`'.$i.'`';}, array_keys($row))) . ')';
    }

    //获取导入sql数据体
    private function createRows($list){
        $arr = [];
        foreach($list as $v){
            $arr[] = $this->implodeRow($v);
        }
        return implode(',', $arr);
    }

    //处理列项上null的情况
    private function implodeRow($row){
        $arr = [];
        foreach($row as $v){
            if(is_null($v)){
                $arr[] = "null";
            }else{
                $v = addslashes($v);
                $arr[] = "'".$v."'";
            }
        }

        return '('.implode(',', $arr).')';
    }
 
    /**
     * 备份表结构
     * @param $talbes
     * @param config 
     *      size_limit      分包大小，如果为0是不分包，如果不为0为分包，大小需要大于1024*1024(即1M)
     *      path            相对路径，相对于dback目录而言的，要加尾部斜杠
     *      name            文件名称，默认为空，默认是包名
     */
    public function backup($tables, array $config=[])
    {
        $size_limit = isset($config['size_limit'])?$config['size_limit']:1024*1024*10;
        if($size_limit > 0 && $size_limit < 1024*1024){
            $size_limit = 1024*1024;        //别太小了,很烦
        }
        $path = isset($config['path'])?$config['path']:time().DIRECTORY_SEPARATOR ;

        $file = [
            'path' => root_path() . "addons" . DIRECTORY_SEPARATOR . "dbassistant" . DIRECTORY_SEPARATOR . "dback" . DIRECTORY_SEPARATOR . $path,      //文件路径
            'name' => isset($config['name'])?$config['name']:"",            //文件名
            'part' => 1,                                                        //当前分包编码
            'size_limit' => $size_limit,                                       //分包文件大小，如果为0就是不分包
            'size' => 0,                                                        //当前文件大小
            'allsize' => 0,                                                     //总大小
            'fp' => null                                                        //文件指针
        ];

        foreach($tables as $table => $conf){
            if(isset($conf['s']) && $conf['s']){
                //导出结构
                $re = $this->tableStructure($file, $table);
                if($re['code'] >= 10100){
                    return $re;
                }
            }
            if(isset($conf['d']) && $conf['d']){
                //导出数据
                $re = $this->tableData($file, $table);
                if($re['code'] >= 10100){
                    return $re;
                }
            }
        }
        $this->close($file);
        return recode(10000, $file, $file['part'].'个文件，'.round($file['allsize']/(1024*1024), 2). "M大小");
    }

    //删除备份
    public function delBackup($path){
        if($path == ""){
            return recode(10100, '', '想啥呢？');
        }
        $path = root_path() . "addons" . DIRECTORY_SEPARATOR . "dbassistant" . DIRECTORY_SEPARATOR . "dback" . DIRECTORY_SEPARATOR . explode(DIRECTORY_SEPARATOR, $path)[0];
        self::deleteDirectory($path);
        return recode(10000);
    }
    private static function deleteDirectory($dir) {
        if (!is_dir($dir)) {
            return false;
        }
        $files = array_diff(scandir($dir), array('.', '..'));
        foreach ($files as $file) {
            (is_dir("$dir/$file")) ? self::deleteDirectory("$dir/$file") : unlink("$dir/$file");
        }
        return rmdir($dir);
    }
 
    /**
     * 导入数据
     * @param integer $start 起始位置
     * @return array|bool|int
     */
    public function import($path)
    {
        $dir = root_path() . "addons" . DIRECTORY_SEPARATOR . "dbassistant" . DIRECTORY_SEPARATOR . "dback" . DIRECTORY_SEPARATOR . $path. DIRECTORY_SEPARATOR. "*.sql";
        $list = glob($dir);

        //Db::startTrans();
        //try {
            foreach($list as $file){
                $gz   = fopen($file, 'r');
                $sql = '';
                do {
                    $sql .= fgets($gz);
                    if (preg_match('/.*;$/', trim($sql))) {
                        Db::execute($sql);
                        // if (false === Db::execute($sql)) {
                        //     return recode(10100, $sql, "执行sql出错了");
                        // }
                        $sql = '';
                    }
                }while(!feof($gz));
                fclose($gz);
            }
            // 提交事务
            //Db::commit();
        // } catch (\Exception $e) {
        //     // 回滚事务
        //     fclose($gz);
        //     Db::rollback();
        //     return recode(10100, '', $e->getMessage());
        // }

        return recode(10001);

    }
 
}